; Elementary drivers for the FRAM "dongle"
; 12 Dec. 1996
; Assembler: Borland Turbo Assembler, V3.1

;=======================================================================;
;                                                                       ;
;   D-X Designs Pty. Ltd.   Perth, Western Australia                    ;
;                                                                       ;
;  Copyright (C) 1996  David R. Brooks                                  ;
;                                                                       ;
;  This program is free software; you can redistribute it and/or        ;
;  modify it under the terms of the GNU General Public License          ;
;  as published by the Free Software Foundation; either version 2       ;
;  of the License, or (at your option) any later version.               ;
;                                                                       ;
;  This program is distributed in the hope that it will be useful,      ;
;  but WITHOUT ANY WARRANTY; without even the implied warranty of       ;
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ;
;  GNU General Public License for more details.                         ;
;                                                                       ;
;  You should have received a copy of the GNU General Public License    ;
;  along with this program; if not, write to the Free Software          ;
;  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            ;
;                                                                       ;
;=======================================================================;


; Standard register usage:
;       AL      Port IO buffer
;       AH      Data shift register
;       BX      FRAM address
;       CX      Loop counter
;       DX      Port address
;       DS:SI   Buffer to write from
;       ES:DI   Buffer to read to

tester  equ     0       ;Define to include self-test code

        .model  small
        .stack  100h
        .code
        .386
        .lall

        ;Entry Hooks

        ifdef   tester
        jmp     testcode
        endif

        jmp     dongle_open
        jmp     dongle_close
        jmp     dongle_read
        jmp     dongle_write

null    equ     0ffffh

pdata   dw      null    ;Data port addr.
pstat   dw      null    ;Status port addr.
pctrl   dw      null    ;Control port addr.

        db      '12 Dec. 1996',0

o_ctl   equ     2       ;Port offset to control reg.
o_stat  equ     1       ;Port offset to status reg.
lpmax   equ     4       ;Max. no. of parallel ports

stb     equ     0       ;STB\ (reset) bit in control reg.
irq     equ     4       ;Interrupt enable ditto
outena  equ     5       ;Output-enable (bi-di ports only)

pe      equ     5       ;PE (data in) bit in status reg.

                ;Bits in data port
lclk    equ     0       ;Dongle-lock clock
scl     equ     1       ;SCL to EEPROM, also lock enable
sda     equ     2       ;SDA to EEPROM (out only)

biosseg equ     40h     ;Segment base for BIOS data
lptab   equ     8       ;Table of LPTx: port numbers

axcnt   equ     15      ;No. of pulses to unlock dongle

dtype   equ     0ah     ;I2C ident. for FM24C04

                ;Return status values (CY don't care)
suxes   equ     0       ;Good
badarg  equ     1       ;Invalid arguments passed
nochip  equ     2       ;Could not locate FRAM device
wrterr  equ     3       ;Lost ACK during write
nodong  equ     4       ;No dongle present
opnerr  equ     5       ;Open/close status incorrect

timeout equ     1000    ;Retry limit for ACK detection
pstim   equ     100     ;Delay after changing signals

maxchip equ     3       ;Highest FRAM address (dongle class)
maxfram equ     200h    ;Address span of FRAM

;---------------------------------------------------

clear   macro   reg,bit ;;Clears 1 bit from reg
        and     reg,not (1 shl bit)
        endm

set     macro   reg,bit ;;Sets 1 bit in reg
        or      reg, 1 shl bit
        endm

pulse   macro   bit     ;;Pulse the given bit high & low
                                ;;NB Too fast for I2C
        out     dx,al           ;;Initial conditions
        set     al,bit
        out     dx,al
        clear   al,bit
        out     dx,al
        endm

;---------------------------------------------------

; B A S I C   I 2 C   R O U T I N E S

;These generally corrupt AX and DX. Other regs. preserved

pause:          ;After changing signals, wait for slow stuff
        push    cx
        mov     cx,pstim
ps1:    loop    ps1
        pop     cx
        ret

sdaget:         ;Get current SDA line to CY
        mov     dx,pstat
        in      al,dx
        sal     al,8-pe         ;Shift it to CY
        ret

sdaset:         ;SDA := 1
        mov     dx,pdata
        in      al,dx
        set     al,sda
        out     dx,al
        call    pause
        ret

sdaclr:         ;SDA := 0
        mov     dx,pdata
        in      al,dx
        clear   al,sda
        out     dx,al
        call    pause
        ret

sdaput:         ;Send CY to the SDA line
        mov     dx,pdata
        pushf
        in      al,dx
        clear   al,sda
        popf                    ;Value in CY
        jnc     sd1
        set     al,sda
sd1:    out     dx,al
        call    pause
        ret

sclset:         ;SCL := 1
        mov     dx,pdata
        in      al,dx
        set     al,scl
        out     dx,al
        call    pause
        ret

sclclr:         ;SCL := 0
        mov     dx,pdata
        in      al,dx
        clear   al,scl
        out     dx,al
        call    pause
        ret

sclclk:         ;Pulse the SCL line
        call    sclset
        call    sclclr
        ret

stop:           ;Send STOP to all devices
        push    ax
        call    sdaclr
        call    sclset
        call    sdaset          ;SDA rises while SCL high
        pop     ax
        ret

start:          ;Send START to all devices
        call    sclset
        call    sdaset
        call    sdaclr          ;SDA falls while SCL high
        call    sclclr
        call    sdaset
        ret

init:           ;Initialise the I2C bus
        call    sdaset
        call    sclclr
        call    stop            ;Be sure all devices reset
        call    stop
        call    stop
        ret

putbyte:        ;AL => I2C bus device
        push    cx
        mov     ah,al
        mov     cx,8
pb1:    sal     ah,1            ;AH[7] => CY
        call    sdaput          ; & so to SDA
        call    sclclk          ;Clock it
        loop    pb1
        call    sdaset          ;Leave SDA high, for ACK
        pop     cx
        ret

getbyte:        ;I2C bus device => AL
        push    cx
        call    sdaset
        mov     cx,8
gb1:    call    sclset          ;Clock UP
        call    sdaget          ;SDA line => CY
        adc     ah,ah           ;Shift into AH
        call    sclclr          ;Clock DOWN
        loop    gb1
        pop     cx
        mov     al,ah           ;Return in AL
        ret

giveack:        ;If CY=0, send ACK; if CY=1, send NACK
        call    sdaput          ;SDA as reqd.
        call    sclclk          ;Clocked through
        call    sdaset
        ret

getack:         ;Get ACK from slave: time out
        push    cx
        call    sdaset
        call    sclset
        mov     cx,timeout
ga1:    call    sdaget
        jnc     short ga2       ;0 - OK
        loop    ga1             ;1 - wait on
ga2:    pushf                   ;CY=1 if timed out
        call    sclclr
        popf
        pop     cx
        ret

logon:          ;Connect to designated slave [AL]
        push    ax
        call    start
        pop     ax
        call    putbyte         ;Send addr.
        call    getack          ;Check it's there
        ret                     ;CY=1 if error

putaddr:        ;AX = address : slave
        push    ax
        call    logon
        pop     ax
        jc      short pa1       ;Failed to connect
        mov     al,ah
        call    putbyte         ;Address
        call    getack
pa1:    ret                     ;Back: CY = error


;---------------------------------------------------

dongle_open:    ;Activate dongle: initial presence check
                ;Pass LPTx: number in AL (1..4)
                ;Errors return CY=1, & code in AX:
                ; badarg  port not available
                ; nodong  no response from dongle
                ; opnerr  port already opened

        cmp     pdata,null
        je      short op3       ;Not opened already
        mov     ax,opnerr       ;Else error
        stc
        ret

op3:    dec     al              ;LPT 0..3
        cmp     al,lpmax
        jc      short op1       ;Range check
faila:  mov     ax,badarg

fail:   stc                     ;Return CY=1 if error
        ret

op1:    mov     ah,0
        shl     ax,1            ;Scale for words
        mov     bx,ax
        push    es
        mov     dx,biosseg
        push    dx
        pop     es
        mov     dx,es:lptab[bx] ;Get port address
        pop     es

        mov     cx,dx
        jcxz    faila           ;Port not assigned

        mov     pdata,dx
        push    dx
        add     dx,o_stat
        mov     pstat,dx
        pop     dx
        add     dx,o_ctl
        mov     pctrl,dx        ;Save values

        mov     dx,pdata
        mov     al,-1
        clear   al,lclk
        clear   al,scl
        out     dx,al           ;LCLK=SCL=0

        mov     ah,al           ;Save it
        mov     dx,pctrl
        in      al,dx
        clear   al,stb
        clear   al,irq          ;No interrupts
        clear   al,outena       ;No bidirectional stuff
        out     dx,al           ;STB\=1 (ie allow unlocking)
        mov     dx,pdata
        mov     al,ah           ;Restore AL, DX

        pulse   lclk            ;This zeros the access counter

        mov     cx,axcnt
        set     al,scl          ;Enable access counter
op2:    pulse   lclk            ;Count once
        loop    op2             ;Unlock the dongle

        call    init            ;Start up the I2C bus

        call    sdaget          ;Check SDA pulled up (dongle)
        cmc                     ;  it should be (ie CY was set)
        mov     ax,nodong
        jc      fail            ;Apparently nothing there
        mov     ax,suxes
        ret                     ;Done: return CY=0 if OK

;---------------------------------------------------

ckopen:                 ;Check dongle is "open"
        cmp     pdata,null
        clc                     ;CY=0, ready
        jne     short cko1
        mov     ax,opnerr       ;No: error
        stc                     ;CY=1
cko1:   ret

;---------------------------------------------------

dongle_close:           ;Deactivate the dongle
                ;Errors return CY=1, & code in AX:
                ; opnerr  port already closed

        call    ckopen
        jnc     short dc0
        ret                     ;Closed already

dc0:    mov     pdata,null      ;Mark it closed
        mov     pstat,null
        mov     pctrl,null

        call    stop            ;Be sure its inactive
        in      al,dx
        clear   al,scl
        pulse   lclk            ;Clear the access counter
        mov     ax,suxes
        clc
        ret

;---------------------------------------------------

ckaddr:         ;Validate calling parameters
        cmp     al,maxchip      ;Valid chip no.
        jnc     short ckf
        push    bx
        add     bx,cx
        cmp     bx,maxfram      ;FRAM addr. in range?
        pop     bx
        ja      short ckf
        jcxz    ckf             ;Length cannot be zero
        add     dx,cx           ;Buffer offset: to end
        jc      short ckf       ;Wrapped over top

        add     al,dtype shl 2  ;Make up I2C address
        shr     bh,1            ;BX[8] => CY
        adc     al,al           ;      => AL[0]
        add     al,al           ;Space for R/W bit in AL[0]
        mov     ah,bl           ;FRAM addr. in AH
        clc                     ;All OK
        ret

ckf:    stc                     ;Errors...
        ret

;---------------------------------------------------

dongle_read:            ;Read data from open dongle
;       AL    = slave no.  (0..3)
;       BX    = FRAM addr. (0..1FF)
;       CX    = length     (0..1FF)
;       ES:DI = buffer

; Errors return CY=1, and code in AX
;  badarg  Invalid FRAM address/length
;  nochip  No ACK received from FRAM chip
;  opnerr  Dongle is not open

        call    ckopen
        jnc     short rd0
        ret                     ;Not open: quit

rd0:    mov     dx,di           ;Offset in DX
        call    ckaddr          ;Check argts.
        jnc     short rd1
        mov     ax,badarg
        jmp     short fini

rd1:    push    ax
        call    putaddr         ;Start as a WRITE: address
        pop     ax
        jnc     short rd2
rd3:    mov     ax,nochip       ;This must be acknowledged
        jmp     short fini

rd2:    or      al,1            ;Set READ bit
        call    logon           ;Re-START, & READ cmnd.
        jc      short rd3       ; failed
        jmp     short rd5

rd4:    clc
        call    giveack         ;ACK after all but the last
rd5:    call    getbyte
        stosb                   ;Data => buffer
        loop    rd4
        stc
        call    giveack         ;NACK at the end
        mov     ax,suxes
        jmp     short fini

;----------------------------------------------------

dongle_write:           ;Write data to dongle
;       AL    = slave no.  (0..3)
;       BX    = FRAM addr. (0..1FF)
;       CX    = length     (0..1FF)
;       DS:SI = buffer

; Errors return CY=1, and code in AX
;  badarg  Invalid FRAM address/length
;  nochip  No ACK received from FRAM chip
;  wrterr  No ACK from write data
;  opnerr  Dongle is not open

        call    ckopen
        jnc     short wr0
        ret                     ;Not open: error

wr0:    mov     dx,si           ;Buffer offset
        call    ckaddr          ;Parameters OK?
        jnc     short wr1
        mov     ax,badarg
        jmp     short fini      ;No!

wr1:    call    putaddr         ;Start the write
        mov     ax,nochip
        jc      short fini      ;Logon failed
wr2:    lodsb
        call    putbyte         ;Write once
        call    getack
        mov     ax,wrterr
        jc      short fini      ;Failed ACK
        loop    wr2
        mov     ax,suxes

fini:   call    stop            ;Done: reset the I2C bus
        cmp     ax,suxes
        jz      short fn2       ;Success: CY=0
        stc                     ;Errors: set CY
fn2:    ret

        ifdef   tester

;===========================================================
;       Self-exerciser routine: run only from DEBUG
;===========================================================

xopen   equ     0ffh            ;Failed to open it
xwrite  equ     0feh            ;Failed write
xread   equ     0fdh            ;Failed read
xcomp   equ     0fch            ;Failed data compare

wdat    db      0,1,2,3,4,5,6,7,8,9,0ah,0bh,0ch,0dh,0eh,0fh
wsize   equ     $-wdat

testcode:
        mov     al,1            ;Use LPT1
        call    dongle_open
        mov     ah,xopen
        jc      short quit

                        ;Write some data
        mov     al,0            ;FRAM identifier
        mov     bx,0            ;FRAM address
        mov     cx,wsize        ;Block length
        mov     si,offset wdat  ;Block address
        push    cs
        pop     ds              ; its segment
        call    dongle_write    ;Do the write
        mov     ah,xwrite
        jc      short quit      ;Failed!

                        ;Read it back
        mov     al,0            ;FRAM identifier
        mov     bx,0            ;FRAM address
        mov     cx,wsize        ;Block size
        sub     sp,wsize        ;Read buffer on stack
        mov     di,sp           ;Read pointer
        push    ss
        pop     es              ; its segt.
        call    dongle_read     ;Do it
        mov     ah,xread
        jc      short quit2     ;Failed

        mov     si,offset wdat
        mov     di,sp           ;Pointers
        mov     cx,wsize
        repe    cmpsb           ;Check the data (ES, DS set up)
        mov     ah,xcomp
        jne     short quit2     ;Fail

        mov     ah,0            ;Success!

quit2:  add     sp,wsize        ;Retrieve stack space

quit:   push    ax
        call    dongle_close    ;AH = step at which stopped
        pop     ax              ;AL = detailed status

        int     3               ;Trap to DEBUG

        endif

        end
